/************************************************************************************
* This file contains the driver primitives for SPI and GPIO interfacing from the
* MAC/PHY to the Radio. The primitives may be implemented as macros or functions
* depending on the speed of the MCU and effectiveness of the compiler. There is a
* potential trade-off between code space and execution speed through this choice.
*
*
* (c) Copyright 2012, Freescale Semiconductor, Inc. All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
************************************************************************************/


/************************************************************************************
*************************************************************************************
* Includes
*************************************************************************************
************************************************************************************/

#include "EmbeddedTypes.h"
#include "TransceiverReg.h"
#include "TransceiverDrv.h"
#include "PortConfig.h"
#include "Interrupt.h"
#include "Utilities_Interface.h"

SPI_MemMapPtr  SpiRegs; 

#define DEFAULT_FRF_915

/*****************************************************************************
******************************************************************************
* Public memory declarations
******************************************************************************
*****************************************************************************/


uint8_t const MKW01DefaultRegisterValues[] = 
{
  /* Radio operation mode initialization @0x01*/
  MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy ) ,
  
  /* Radio Data mode and modulation initialization @0x02*/
  MKW01_Reg_DataModul, (uint8_t) ( DataModul_DataMode_Packet | DataModul_Modulation_Fsk | DataModul_ModulationShaping_BT_05) ,

  /* Radio bit rate initialization @0x03-0x04*/
  MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_4800 ) ,
  MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_4800 ) ,
  
  /* Radio frequency deviation initialization @0x05-0x06*/
  MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_2400 ) ,
  MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_2400 ) ,
  
  /* Radio RF frequency initialization @0x07-0x09*/
  /*@CMA, Default Frequencies*/
#ifdef DEFAULT_FRF_915
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_915 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_915 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_915 ) ,
#endif

#ifdef DEFAULT_FRF_868  
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_868 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_868 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_868 ) ,
#endif

#ifdef DEFAULT_FRF_865
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_865 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_865 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_865 ) ,
#endif

#ifdef DEFAULT_FRF_470  
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_470 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_470 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_470 ) ,
#endif

#ifdef DEFAULT_FRF_434  
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_434 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_434 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_434 ) ,
#endif  

#ifdef DEFAULT_FRF_920 
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_920 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_920 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_920 ) ,
#endif  
  /* Radio RegAfcCtrl initialization @0x0B*/
  MKW01_Reg_AfcCtrl, (uint8_t) AfcCtrl_AfcLowBeta_Off ,
  
  /* Radio output power initialization @0x11*/
  MKW01_Reg_PaLevel, (uint8_t) ( PaLevel_Pa0_On | PaLevel_Pa1_Off | PaLevel_Pa2_Off | 0x1F ) ,
  
  /* Radio Rise/Fall time of ramp up/down in FSK initialization @0x12*/
  MKW01_Reg_PaRamp, (uint8_t) (PaRamp_40) ,
  
  /* Radio overload current protection for PA initialization 0x13*/
  MKW01_Reg_Ocp, (uint8_t) ( Ocp_Ocp_On | 0x0C) ,
  
  /* Radio LNA gain and input impedance initialization @0x18*/
  MKW01_Reg_Lna, (uint8_t) ( Lna_LnaZin_50 | Lna_LnaGain_Agc ) ,
  
  /* Radio channel filter bandwidth initialization @0x19*/
  MKW01_Reg_RxBw, ( DccFreq_7 | RxBw_10400 ) ,
  
  /* Radio channel filter bandwidth for AFC operation initialization @0x1A*/
  MKW01_Reg_AfcBw, ( DccFreq_7 | RxBw_10400 ) ,
  
  /* Radio automatic frequency control initialization @0x1E*/
  MKW01_Reg_AfcFei, (uint8_t) ( AfcFei_AfcAuto_Off | AfcFei_AfcAutoClear_On ) ,

  /* Radio Rssi threshold initialization @0x29*/
  // RSSIthres = [-174 + NF +10*log(2*RxBw) + DemodSNR] dBm
  // NF = 7dB
  // DemodSnr = 8dB
  // RxBw depends on frequency bands and profiles
  MKW01_Reg_RssiThresh, (uint8_t) (0xDC) , // -101 dBm for 333.3 Khz singleside channel filter bandwith
  
  /* Radio RegTimeoutRxStart initialization @0x2A*/
  /* Radio RegTimeoutRssiThresh initialization @0x2B*/
  MKW01_Reg_RxTimeout1, (uint8_t) 0x00 , //disable timeout rx start
  MKW01_Reg_RxTimeout2, (uint8_t) 0x00 , //disable timeout rx start
  
  /* MKW01 preamble size initialization @0x2C-0x2D*/
  MKW01_Reg_PreambleMsb, (uint8_t) 0x00,
  MKW01_Reg_PreambleLsb, (uint8_t) 0x10,
  
  /* Radio sync word control and value initialization @0x2E-0x30*/
  MKW01_Reg_SyncConfig, (uint8_t) (SyncConfig_Sync_On | SyncConfig_FifioFill_ifSyncAddres | SyncConfig_SyncSize_2) ,
  MKW01_Reg_SyncValue1, (uint8_t) (0x90) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue2, (uint8_t) (0x4E) , //SFD value for uncoded with phySUNMRFSKSFD = 0

  /* Radio packet mode config */
  
  MKW01_Reg_PacketConfig1, 0xD8,
  MKW01_Reg_PacketConfig2, 0x00,
  
  /* Radio payload length initialization */
  MKW01_Reg_PayloadLength, (uint8_t) 255  //max length in rx
};

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/


/******************************************************************************
* Description :SPI initialization for Radio driver interface
*               
* Assumptions : 
*               
* Inputs      :none 
* Output      :none 
* Errors      :none 
******************************************************************************/
void MKW01Drv_SpiInit(void)
{
   volatile uint8_t dummy;
   
        /* Enable Reset PORT clock */   
  MKW01x_RESET_PORT_SIM_SCG |= cRESET_PORT_SIM_SCG_Config_c;

     /* Enable Reset */ 
  gMKW01RadioResetPort  &= ~PORT_PCR_MUX_MASK;
  gMKW01RadioResetPort  |= PORT_PCR_MUX(1);
  gRadioReset_DataAvailablePortDDirReg_c|=gMKW01ResetMask_c;
     
      /* Enable SPI PORT clock */   
  MKW01x_SPI_PORT_SIM_SCG |= cSPI_PORT_SIM_SCG_Config_c;
  
    /* Enable SPI clock */ 
  
  MKW01x_SPI_SIM_SCG|= cSPI_SIM_SCG_Config_c;
  
    /* Initialize SPI Pin Control registers */
  SPI_PCS_PCR  &= ~PORT_PCR_MUX_MASK;
  SPI_PCS_PCR  |= PORT_PCR_MUX(1);                                    
  gSPI_DataAvailablePortDDirReg_c|= gSPI_DataAvailablePinMask_c;      
  SPI_SCK_PCR  &= ~PORT_PCR_MUX_MASK;
  SPI_SCK_PCR  |= PORT_PCR_MUX(SPI_SCK_ALT_FUNC);
  SPI_MOSI_PCR &= ~PORT_PCR_MUX_MASK;
  SPI_MOSI_PCR |= PORT_PCR_MUX(SPI_MOSI_ALT_FUNC);
  SPI_MISO_PCR &= ~PORT_PCR_MUX_MASK;
  SPI_MISO_PCR |= PORT_PCR_MUX(SPI_MOSI_ALT_FUNC);   
  
  MKW01Drv_AssertReset(); //Keep Radio in reset
  
  /* Initialize the SPI peripheral base address */
  SpiRegs = SPI0_BASE_PTR;  
  SpiRegs->C1 = cSPIC1_Init;
  SpiRegs->C2 = cSPIC2_Init;
  SPI0_BR = cSPIBR_Init;
  SpiRegs->C1 |= (uint8_t)(SPI_C1_SPE_MASK | SPI_C1_MSTR_MASK) ; //enable SPI system
  SPIS_DummyRead;  
  DeAssertSS                                                          
 
}

/******************************************************************************
* Description :Make a pulse on Radio hardware reset PIN
*               
* Assumptions : 
*               
* Inputs      :none 
* Output      :none 
* Errors      :none 
******************************************************************************/
bool_t MKW01Drv_Reset(void)
{
  //@AC RESET sequence from SX1233 datasheet:
  // RESET high Z
  // RESET = 1, then wait 100us
  // RESET = 0 (high Z), then wait 5ms before using the Radio. 
  
  MKW01Drv_AssertReset();
  Delay_us(100);                                                                //current delay is 230uS? TODO, Remeasure
  MKW01Drv_DeassertReset();  
  Delay_ms(5);                                                                  //current delay is 15.6mS? TODO, Remeasure

  while((gMKW01ClkOutPort & gMKW01ClkOutMask_c) == 0);                          //Need to wait for CLKOUT to be enabled.
  return TRUE;
  
}

/******************************************************************************
* Description :HardCore MK01 Delay Helper Functions                        
*               
* Assumptions : 
*               
* Inputs      :none 
* Output      :none 
* Errors      :none 
******************************************************************************/
void Wait_6_BitClks(void)       // Duration : BRA/BSR/JSR to here: 3/5/6 mcuCycles      //TODO!! Take off or calculate accordingly for KL25 core
{                               //            2xNOP+RTS          :     8 mcuCycles
  asm ("nop");
  asm ("nop");
}

void Wait_8_BitClks(void)                                                               //TODO!! Take off or calculate accordingly for KL25 core
{                               // Duration : BSR/JSR to here: 5/6 mcuCycles
  Wait_6_BitClks();             // Uses BRA (3 mcuCyc) to above function (so it really only takes 5,5 bitclks)
}




/*************************************************************************
*                       SPI Read function                                *
*************************************************************************/

/******************************************************************************
* Description :Read one 8bit data-location from MKW01 register "addr"
*              Unprotected - only to be called with disabled interrupts
* 
* Assumptions : 
*               
* Inputs      :register address
* Output      :register value 
* Errors      :none 
******************************************************************************/
uint8_t MKW01Drv_ReadRegisterFast(uint8_t addr) {
  uint8_t val;

  SPIS_DummyRead;
  AssertSS;
  SpiRegs->D = (unsigned char)(addr);                                           // Write Addr byte  
  WaitSPITxBufferEmpty; 
  SPID_DummyRead;
  SpiRegs->D = 0x00;          /* Write data byte */
  WaitSPITxBufferEmpty;                                                         // Wait for room in holding register 
  SPID_DummyRead;                             
  WaitSPIRxBufferFull;                                                          //Wait until the shift buffer has received the data from the Slave, this ensures that the Tx Data has been sent and the Slave has received it
  val = SpiRegs->D;          /* Write data byte */
  DeAssertSS;
  return val;
}

/******************************************************************************
* Description :Read one 8bit data-location from Radio register "addr"
*              May be called from any context.
*              Protected from mcu/app irq. Includes irq status backup
* Assumptions : 
*               
* Inputs      :register address
* Output      :register value 
* Errors      :none 
******************************************************************************/
uint8_t MKW01Drv_ReadRegister(uint8_t addr) {
  uint8_t val;
  uint32_t CCR;	      

#if defined(IAR)	
  CCR = IntDisableAll();	
#elif defined (CW)
  IntDisableAll(CCR);		
#endif

  SPIS_DummyRead;
  AssertSS;
  SpiRegs->D = (unsigned char)(addr);                   // Write Addr byte  
  WaitSPITxBufferEmpty; 
  SPID_DummyRead;
  SpiRegs->D = 0x00;          /* Write data byte */
  WaitSPITxBufferEmpty;                                                         // Wait for room in holding register 
  SPID_DummyRead;                              
  WaitSPIRxBufferFull;                                                          //Wait until the shift buffer has received the data from the Slave, this ensures that the Tx Data has been sent and the Slave has received it
  val = SpiRegs->D;          /* Write data byte */
  DeAssertSS;
  IntRestoreAll(CCR);  
  return val;
}


/*************************************************************************
*                      SPI Write functions                               *
*************************************************************************/

/******************************************************************************
* Description :Write one 8bit data into Radio at address "addr"
*              Unprotected - only to be called with disabled interrupts          
*  
* Assumptions : 
*               
* Inputs      :register address, register value
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_WriteRegisterFast(uint8_t addr, uint8_t val) {
  SPIS_DummyRead;                               
  AssertSS;
  SpiRegs->D = (unsigned char)(addr | 0x80);  /* Write Addr byte */
  WaitSPITxBufferEmpty;                        /* Wait for room in holding register */
  SPID_DummyRead;
  SpiRegs->D = (unsigned char)(val);          /* Write data byte */
  WaitSPITxBufferEmpty;
  SPID_DummyRead;                        
  WaitSPIRxBufferFull;                                                          //Wait for Reg buffer full, this ensures Tx was finished and the Slave received the package, before bringing up the Chip Select line
  DeAssertSS;
}

/******************************************************************************
* Description :Write one 8bit data into Radio at address "addr"
*              May be called from any context.
*              Protected from mcu/app irq. Includes irq status backup
* Assumptions : 
*               
* Inputs      :register address, register value
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_WriteRegister(uint8_t addr, uint8_t val) {
		
  uint32_t CCR;

#if defined(IAR)	
  CCR = IntDisableAll();	
#elif defined (CW)
  IntDisableAll(CCR);		
#endif
  
  SPIS_DummyRead;
  AssertSS;                                   
  SpiRegs->D = (unsigned char)(addr | 0x80);  /* Write Addr byte */
  WaitSPITxBufferEmpty;                        /* Wait for room in holding register */
  SPID_DummyRead;
  SpiRegs->D = (unsigned char)(val);          /* Write data byte */
  WaitSPITxBufferEmpty;                                                         // Wait for room in holding register 
  SPID_DummyRead;                              
  WaitSPIRxBufferFull;                                                          //Wait until the shift buffer has received the data from the Slave, this ensures that the Tx Data has been sent and the Slave has received it
  DeAssertSS;                          
  IntRestoreAll(CCR);
}

/******************************************************************************
* Description :Initialize Radio transceiver with default registers value
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_RFdefaultInit(void)
{
  uint8_t idx;
  for(idx = 0; idx < (sizeof(MKW01DefaultRegisterValues)/sizeof(MKW01DefaultRegisterValues[0])); idx += 2)
  {
    MKW01Drv_WriteRegister(MKW01DefaultRegisterValues[idx], MKW01DefaultRegisterValues[idx+1]);
  }
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_ConfigureDIOPinsFast(uint8_t regDio1, uint8_t regDio2)
{
  uint8_t reg2;
  MKW01Drv_WriteRegisterFastMacro(MKW01_Reg_DioMapping1, regDio1);
  MKW01Drv_ReadRegisterFastMacro(MKW01_Reg_DioMapping2, reg2);
  reg2 = (uint8_t) (reg2 & 0x07);
  reg2 = (uint8_t) (reg2 | (regDio2 & 0xF0) );
  MKW01Drv_WriteRegisterFastMacro(MKW01_Reg_DioMapping2, reg2);
   
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_ConfigureDIOPins(uint8_t regDio1, uint8_t regDio2)
{
  uint8_t reg2;
  MKW01Drv_WriteRegister(MKW01_Reg_DioMapping1, regDio1);
  reg2 = MKW01Drv_ReadRegister(MKW01_Reg_DioMapping2);
  reg2 = (uint8_t) (reg2 & 0x07);
  reg2 = (uint8_t) (reg2 | (regDio2 & 0xF0) );
  MKW01Drv_WriteRegister(MKW01_Reg_DioMapping2, reg2);
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_ConfigureCLKO(uint8_t fxOscDiv)
{
  uint8_t regDio2;
  regDio2 = MKW01Drv_ReadRegister(MKW01_Reg_DioMapping2);
  regDio2 = (uint8_t)(regDio2 & 0xF8);
  MKW01Drv_WriteRegister(MKW01_Reg_DioMapping2, (uint8_t) (regDio2 | fxOscDiv));

}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_SetOperatingMode(uint8_t mode)
{
  uint8_t regOpMode;
  regOpMode = MKW01Drv_ReadRegister(MKW01_Reg_OpMode);
  regOpMode = (regOpMode & 0xE3) | mode;
  MKW01Drv_WriteRegister(MKW01_Reg_OpMode, regOpMode);
  while ((MKW01Drv_ReadRegister(MKW01_Reg_IrqFlags1) & IrqFlags1_ModeReady) == 0x00);
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
uint8_t MKW01Drv_RadioStandByReq(uint8_t clkoEn)
{
  (void) clkoEn;
  MKW01Drv_SetOperatingMode(OpMode_StandBy);
  return 0;
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
uint8_t MKW01Drv_RadioSleepReq(void)
{
  MKW01Drv_SetOperatingMode(OpMode_Sleep);
  return 0;
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_RadioWakeUpReq(void)
{
  MKW01Drv_SetOperatingMode(OpMode_StandBy);
}

/******************************************************************************
* Description :
*              
* Assumptions : 
*               
* Inputs      :none
* Output      :none
* Errors      :none 
******************************************************************************/
void MKW01Drv_RC_Calibration(void)                                              
{
  volatile uint8_t dummy;
  dummy = MKW01Drv_ReadRegister(MKW01_Reg_Version);
  /*if chip version is V2a perform RC calibraton once at POR*/
  if(MKW01Drv_ReadRegister(MKW01_Reg_Version) == 0x21)                          
  {                                                                             
    MKW01Drv_WriteRegister(MKW01_Reg_OpMode, 0x04);                             //set transceiver in StandBy mode
    
    MKW01Drv_WriteRegister(0x57, 0x80);
    MKW01Drv_WriteRegister(MKW01_Reg_Osc1, (MKW01Drv_ReadRegister(MKW01_Reg_Osc1) | 0x80));
    while((MKW01Drv_ReadRegister(0x0A) & 0x40) == 0x00);
    MKW01Drv_WriteRegister(MKW01_Reg_Osc1, (MKW01Drv_ReadRegister(MKW01_Reg_Osc1) | 0x80));
    while((MKW01Drv_ReadRegister(MKW01_Reg_Osc1) & 0x40) == 0x00);
    MKW01Drv_WriteRegister(0x57, 0x00);
  }
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_PortConfig
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01xDrv_IRQ_PortConfig
(
  void
)
{
  //Enable PORTD interrupts
  NVIC_ICPR |= 0x80000000;
  NVIC_ISER |= 0x80000000;
  MKW01_IRQ_PORT_SIM_SCG |= cIRQ_PORT_SIM_SCG_Config_c;
  MKW01_IRQ_DIO0_PCR |= cIRQ_PCR_Config_c;
  MKW01_IRQ_DIO1_PCR |= cIRQ_PCR_Config_c;
  MKW01_IRQ_DIO4_PCR |= cIRQ_PCR_Config_c;
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_Enable
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01Drv_IRQ_Enable
(uint8_t SensitivityDIO0, uint8_t SensitivityDIO1
)
{

   MKW01_IRQ_DIO0_PCR |= PORT_PCR_ISF_MASK; // Write logic 1 for cleaning the interrupt flag.
   MKW01_IRQ_DIO0_PCR |= PORT_PCR_IRQC(SensitivityDIO0);  //Enable the interrupt when with the desired sensitivity
   MKW01_IRQ_DIO1_PCR |= PORT_PCR_ISF_MASK; // Write logic 1 for cleaning the interrupt flag.   
   MKW01_IRQ_DIO1_PCR |= PORT_PCR_IRQC(SensitivityDIO1);  //Enable the interrupt when with the desired sensitivity
}


/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_Clear
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01Drv_CCA_Enable
(
void
)
{

   MKW01_IRQ_DIO4_PCR |= PORT_PCR_ISF_MASK; // Write logic 1 for cleaning the interrupt flag.
   MKW01_IRQ_DIO4_PCR |= PORT_PCR_IRQC(9);  //Enable the interrupt when rising edge occurs      
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_Clear
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01Drv_IRQ_Clear
(
  void
)
{

  MKW01_IRQ_DIO0_PCR |= PORT_PCR_ISF_MASK;
  MKW01_IRQ_DIO1_PCR |= PORT_PCR_ISF_MASK;
 
}


/*---------------------------------------------------------------------------
 * Name: MKW01Drv_CCA_Clear
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01Drv_CCA_Clear
(
  void
)
{

  MKW01_IRQ_DIO4_PCR |= PORT_PCR_ISF_MASK;
 
 
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_Disable
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01Drv_IRQ_Disable
(
void
)
{
  MKW01_IRQ_DIO0_PCR &= ~PORT_PCR_IRQC_MASK;
  MKW01_IRQ_DIO1_PCR &= ~PORT_PCR_IRQC_MASK;

 
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_DIO1_Disable
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/

void MKW01Drv_IRQ_DIO1_Enable                                                   //TODO! Need to enable Sensitivity parameter
(
void
)
{
   MKW01_IRQ_DIO1_PCR |= PORT_PCR_ISF_MASK;                                     // Write logic 1 for cleaning the interrupt flag.   
   MKW01_IRQ_DIO1_PCR |= PORT_PCR_IRQC(9);                                      //Enable the interrupt when rising edge occurs  
}

void MKW01Drv_IRQ_DIO1_Disable
(
void
)
{
  MKW01_IRQ_DIO1_PCR &= ~PORT_PCR_IRQC_MASK;

 
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_DIO0_Disable
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/

void MKW01Drv_IRQ_DIO0_Enable                                                   //TODO! Need to enable Sensitivity parameter
(
void
)
{
   MKW01_IRQ_DIO0_PCR |= PORT_PCR_ISF_MASK;                                     // Write logic 1 for cleaning the interrupt flag.
   MKW01_IRQ_DIO0_PCR |= PORT_PCR_IRQC(9);                                      //Enable the interrupt when rising edge occurs    

}

void MKW01Drv_IRQ_DIO0_Disable
(
void
)
{
  MKW01_IRQ_DIO0_PCR &= ~PORT_PCR_IRQC_MASK;
 
}

/*---------------------------------------------------------------------------
 * Name: MKW01Drv_IRQ_Disable
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void MKW01Drv_CCA_Disable
(
void
)
{
  MKW01_IRQ_DIO4_PCR &= ~PORT_PCR_IRQC_MASK; 
}


/*---------------------------------------------------------------------------
 * Name: PhyRadio_ISR
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PhyRadio_ISR                                                               
(
  void
)
{
  if(MKW01_IRQ_DIO0_PCR & PORT_PCR_ISF_MASK )
  {
    
  }
  else if (MKW01_IRQ_DIO1_PCR & PORT_PCR_ISF_MASK)
  {
    
  }
}
